#ifndef MATRIX_HPP
#define MATRIX_HPP

#ifndef HANDLEXCEPTION
#define HANDLEXCEPTION 0
#endif

/*!
    \file matrix.hpp
    \brief Base matrix classes

    The file defines the base matrix classes:
    abstract class Matrix with row/column view,
    classes of MatrixSum, MatrixSub to get view of sum or substract of matrices,
    class of ReallocatableMatrix which stored by columns and
    class of function-generated matrix.
*/

//! A abstract class for vectors
/*! This class is used for get view of matrix rows and cols
 */
class Vector {
public:
    //! The constructor
    Vector() { }
    //! The destructor
    virtual ~Vector() { }
    //! Get element in position i
    virtual double elem(unsigned i) const = 0;
    //! Get size of vector
    virtual unsigned GetSize() const = 0;
};

//! A abstract class for matrices
class Matrix {
public:
    //! The constructor
    Matrix() { }
    //! The destructor
    virtual ~Matrix() { }
    //! Get number of rows
    virtual unsigned GetNumRows() const = 0;
    //! Get number of columns
    virtual unsigned GetNumCols() const = 0;
    //! Get element in position i,j
    virtual double elem(unsigned i, unsigned j) const = 0;

    //! Class of view matrix row
    /*! This class contains of pointer to matrix to get usefull acess to
     *  its row
     */
    class Row : public Vector {
        friend class Matrix;
        const Matrix* M;
        const unsigned i;
        Row(const Matrix* _M, unsigned _i);
    public:
        double elem(unsigned j) const;
        unsigned GetSize() const;
    };
    //! Get view of i row of matrix
    Row GetRow(unsigned i) const;
    //! Class of view matrix column
    /*! This class contains of pointer to matrix to get usefull acess to
     *  its column
     */
    class Column : public Vector {
        friend class Matrix;
        const Matrix* M;
        const unsigned j;
        Column(const Matrix* _M, unsigned _j);
    public:
        double elem(unsigned i) const;
        unsigned GetSize() const;
    };
    //! Get view of j column of matrix
    Column GetCol(unsigned j) const;
};


//! Abstract class of pair matrix with equal sizes; TODO: rename this class
class MatrixBinExpr : public Matrix {
protected:
    const Matrix& M1;
    const Matrix& M2;
public:
    //! Create pair of matrices M1 and M2 which sizes are equal
    MatrixBinExpr(const Matrix& _M1, const Matrix& _M2);
    // TODO: remove next comment
    // Matrix Calc() const; write calculated expr in matrix
    //! Get first matrix from pair
    const Matrix& GetFirst() const;
    //! Get second matrix from pair
    const Matrix& GetSecond() const;
    unsigned GetNumRows() const;
    unsigned GetNumCols() const;
};

//! Class of view of sum of two matrices
class MatrixSum : public MatrixBinExpr {
public:
    //! Create view of sum of two matrices
    MatrixSum(const Matrix& _M1, const Matrix& _M2);
    double elem(unsigned i, unsigned j) const;
};

//! Class of view of substraction of two matrices
class MatrixSub : public MatrixBinExpr {
public:
    //! Create view of substraction of two matrices
    MatrixSub(const Matrix& _M1, const Matrix& _M2);
    double elem(unsigned i, unsigned j) const;
};

//! Number of size which added to size of columns of ReallocatableMatrix
enum {ReallocatableMatrix_add_allocsize = 10};

//! Class of matrix stored by columns
/*! This class contains matrix which stored by columns so you can easy
 *  add columns.
 */
class ReallocatableMatrix : public Matrix {
    unsigned num_rows;
    unsigned num_cols;
    unsigned allocsize;
    double **mat;
public:
    //! Initialize matrix with sizes rows x start_cols
    ReallocatableMatrix(unsigned rows, unsigned start_cols);
    unsigned GetNumRows() const;
    unsigned GetNumCols() const;
    //! Add new columns two matrix if needed
    void Resize(unsigned needsize);
    //! Get a reference to matrix in position i,j; so you can change it
    double& elem(unsigned i, unsigned j);
    double elem(unsigned i, unsigned j) const;
    //! The destructor
    ~ReallocatableMatrix();
};

//! Class of matrix generated by function
class FunctionMatrix : public Matrix {
    unsigned num_rows;
    unsigned num_cols;
    double h;
    double (*f)(unsigned, unsigned, double);
public:
    //! Initialize matrix wich i,j element equal func(i,j,h)
    FunctionMatrix(unsigned rows, unsigned cols, double _h,
            double (*func)(unsigned, unsigned, double));
    unsigned GetNumRows() const;
    unsigned GetNumCols() const;
    double elem(unsigned i, unsigned j) const;
};

#endif /* MATRIX_HPP */
